# -*- coding: utf-8 -*-
from PyQt5.Qt import Qt, QWidget, QDialog, QPaintEvent, QPainter, QPainterPath, QPixmap, QColor, QTimer,  QIcon, \
    QMouseEvent, QKeyEvent, QGraphicsDropShadowEffect
from daemon.component.layout import VBoxLayout
from daemon.component.title_bar import TitleBar
from common import filesystem


class BaseDialog(QDialog):

    def __init__(self, title: str, parent: QWidget = None):
        """对话框UI
        """

        super(BaseDialog, self).__init__(parent)
        self.setWindowTitle(title)
        self.setWindowIcon(QIcon(filesystem.ic_logo))

        # 鼠标点击坐标变量
        self.__mouse_press_pos = None
        # 是否点击了确定按钮
        self.confirm_clicked = False

        # 无边框透明
        self.setWindowFlags(Qt.FramelessWindowHint | Qt.Dialog | Qt.WindowCloseButtonHint)
        self.setAttribute(Qt.WA_TranslucentBackground, True)

        # 阴影
        shadow = QGraphicsDropShadowEffect()
        shadow.setOffset(0, 0)
        shadow.setColor(QColor('#666666'))
        shadow.setBlurRadius(8)
        self.setGraphicsEffect(shadow)

        # 淡出淡入
        self.setWindowOpacity(0)
        self.opacity_timer = QTimer()
        self.opacity_timer.timeout.connect(self.__handle_opacity_timer_timeout)
        self.opacity_timer.start(20)

        # 窗口布局
        self.title_bar = TitleBar(title, minimizable=False)
        self.body_widget = self.init_body()

        main_layout = VBoxLayout()
        main_layout.setContentsMargins(26, 10, 20, 10)
        main_layout.addWidget(self.title_bar)
        main_layout.addSpacing(8)
        main_layout.addWidget(self.body_widget)
        self.setLayout(main_layout)

    def init_body(self) -> QWidget:
        """子类必须重写，初始化窗口主体
        """

        pass

    def paintEvent(self, event: QPaintEvent):
        """绘制界面背景
        """

        # 抗锯齿画笔
        painter = QPainter(self)
        painter.setRenderHint(QPainter.Antialiasing, True)

        # 圆角矩形边框
        bg_round_rect_path = QPainterPath()
        bg_round_rect_path.addRoundedRect(10, 10, self.width() - 20, self.height() - 20, 2, 2)
        painter.setClipPath(bg_round_rect_path)

        # 背景
        bg_pixmap = QPixmap(self.width(), self.height())
        bg_pixmap.fill(QColor(255, 255, 255))
        painter.drawPixmap(10, 10, self.width() - 20, self.height() - 20, bg_pixmap)

    def mousePressEvent(self, event: QMouseEvent):
        """鼠标点击事件，记录鼠标按下位置
        """

        if event.buttons() == Qt.LeftButton and self.title_bar.geometry().contains(event.pos()):
            self.__mouse_press_pos = event.globalPos() - self.pos()

    def mouseMoveEvent(self, event: QMouseEvent):
        """鼠标移动事件，拖动窗口
        """

        if self.__mouse_press_pos is not None:
            self.setCursor(Qt.SizeAllCursor)
            self.move(event.globalPos() - self.__mouse_press_pos)

    def mouseReleaseEvent(self, event: QMouseEvent):
        """鼠标释放事件，停止拖动
        """

        self.__mouse_press_pos = None
        self.setCursor(Qt.ArrowCursor)

    def keyPressEvent(self, event: QKeyEvent):
        """重写回车键和ESC行为
        """

        if event.key() == Qt.Key_Escape:
            self.close()
        elif event.key() == Qt.Key_Return or event.key() == Qt.Key_Enter:
            self.confirm()

    def exec_(self) -> bool:
        """重写exec方法，返回是否点击了确定按钮
        """

        super(BaseDialog, self).exec_()
        return self.confirm_clicked

    def confirm(self):
        """确定按钮点击事件，需要被子类重写
        """

        self.confirm_clicked = True
        self.close()

    def __handle_opacity_timer_timeout(self):
        """淡出淡入设置透明度
        """

        if self.windowOpacity() < 0.9:
            self.setWindowOpacity(self.windowOpacity() + 0.05)
        else:
            self.opacity_timer.stop()
