# -*- coding: utf-8 -*-
from PyQt5.Qt import Qt, QWidget, QLabel, QPaintEvent, QPainter, QPainterPath, QColor, QEvent, QPixmap, \
    QMouseEvent, pyqtSignal
from daemon.helper import ui_helper


class ClickableLabel(QLabel):

    clicked = pyqtSignal()

    def __init__(self, *__args):
        """响应点击事件的QLabel
        """

        super(ClickableLabel, self).__init__(*__args)
        self.setCursor(Qt.PointingHandCursor)

    def mouseReleaseEvent(self, event: QMouseEvent):
        """重载鼠标点击事件，发送信号
        """

        if event.button() == Qt.LeftButton and self.geometry().contains(event.pos() + self.pos()):
            self.clicked.emit()


class Button(QWidget):

    clicked = pyqtSignal()

    def __init__(self, text: str, width: int, height: int, parent: QWidget = None, font_size: int = None):
        """按钮
        """

        super(Button, self).__init__(parent)
        self.__is_active = False
        self.setFixedSize(width, height)
        self.label = QLabel(text, self)
        ui_helper.attr(self.label, width=width, height=height, font_size=font_size, align=Qt.AlignCenter)
        self.setCursor(Qt.PointingHandCursor)

    def paintEvent(self, event: QPaintEvent):
        """绘制背景
        """

        # 创建画笔
        painter = QPainter(self)
        # 绘制边框
        border_color = QColor(154, 203, 222) if self.__is_active else QColor(153, 153, 153)
        painter.setPen(border_color)
        painter.drawRect(0, 0, self.width() - 1, self.height() - 1)
        bg_rect_path = QPainterPath()
        bg_rect_path.addRect(0, 0, self.width() - 1, self.height() - 1)
        painter.setClipPath(bg_rect_path)
        # 绘制背景
        bg_color = QColor(243, 249, 251) if self.__is_active else QColor(255, 255, 255)
        bg_pixmap = QPixmap(self.width() - 4, self.height() - 4)
        bg_pixmap.fill(bg_color)
        painter.drawPixmap(2, 1, self.width() - 1, self.height() - 1, bg_pixmap)

    def mousePressEvent(self, event: QMouseEvent):
        """鼠标按下事件
        """

        if event.buttons() == Qt.LeftButton:
            self.label.move(1, 1)

    def mouseReleaseEvent(self, event: QMouseEvent):
        """鼠标释放事件
        """

        self.label.move(0, 0)
        if self.geometry().contains(event.pos() + self.pos()):
            self.clicked.emit()

    def enterEvent(self, event: QEvent):
        """鼠标进入事件
        """

        self.__is_active = True
        self.repaint()

    def leaveEvent(self, event: QEvent):
        """鼠标离开事件
        """

        self.__is_active = False
        self.repaint()
