# -*- coding: utf-8 -*-
import base64
from PyQt5.Qt import Qt, QWidget, QPainter, QPainterPath, QColor, QPixmap, QApplication, QFontMetrics, QFileDialog, \
    QImage, QDesktopWidget, QSize, QTranslator
from common import filesystem

# 缩放系数
__scale_rate = None
# 窗口尺寸
__screen_size = None
# 全局加载中对话框
loading_dialog = None


def attr(widget: QWidget, text: str = None, font_size: int = None, color: str = None, bg_color: str = None,
         border_color: str = None, radius: int = None, padding: int = None, x: int = None, y: int = None,
         width: int = None, height: int = None, align: int = None, text_line: int = 1, text_cut: bool = False,
         image_path: str = None, image_b64: str = None, other_style: str = None, clicked=None) -> QWidget:
    """设置QWidget的属性：文本内容、字体大小、颜色，在父容器的xy坐标等

    :param widget: QWidget控件，如QLabel、QTextEdit、QLineEdit等
    :param text: 文本内容，适用于QLabel等
    :param font_size: 字体大小，适用于QLabel等
    :param color: 文字颜色，适用QLabel等，如#333333
    :param bg_color: 背景颜色，适用QLabel等，如#333333
    :param border_color: 边框颜色
    :param radius: 圆角
    :param padding: 内边距
    :param x: x坐标位移，等同于调用move方法
    :param y: y坐标位移，等同于调用move方法
    :param width: 宽度
    :param height: 高度
    :param align: 对齐方式，适用于QLabel等
    :param text_line: 文字行数，默认1行
    :param text_cut: 文字是否裁剪，超出部分用省略号代替，默认不裁剪
    :param image_path: 图片路径，适用于图片QLabel
    :param image_b64: 图片base64，适用于图片QLabel
    :param other_style: 其他样式属性
    :param clicked: 点击事件的回调方法，适用于QPushButton等f()
    """

    # QSS样式字符串
    style_sheet = ''
    # 文本内容
    if text:
        widget.setText(text)
    # 字体大小
    if font_size:
        font = widget.font()
        font.setPointSize(font_size / scale_rate())
        widget.setFont(font)
    # 颜色
    if color:
        style_sheet += 'color: {};'.format(color)
    # 背景颜色
    if bg_color:
        style_sheet += 'background: {};'.format(bg_color)
    # 边框颜色
    if border_color:
        style_sheet += 'border: 1px solid {};'.format(border_color)
    else:
        style_sheet += 'border: 0;'
    # 圆角
    if radius:
        style_sheet += 'border-radius: {}px;'.format(radius)
    # 内边距
    if padding:
        style_sheet += 'padding: {}px;'.format(padding)
    # 其他样式
    if other_style:
        style_sheet += other_style
    # QSS样式
    if style_sheet != '':
        widget.setStyleSheet(style_sheet)
    # 坐标位移
    if x:
        widget.move(x, widget.y())
    if y:
        widget.move(widget.x(), y)
    # 宽度和高度
    if width:
        widget.setFixedWidth(width)
    if height:
        widget.setFixedHeight(height)
    # 对齐方式
    if align:
        widget.setAlignment(align)
    # 文字换行（自动裁剪）
    if text and text_line > 1:
        __word_wrap(widget, text, text_line)
    # 文字裁剪（仅当不换行时判断是否需要裁剪）
    if text and text_line == 1 and text_cut:
        widget.setText(QFontMetrics(widget.font()).elidedText(text, Qt.ElideRight, widget.width() * text_line))
    # 图片路径
    if image_path:
        widget.setPixmap(QPixmap(image_path))
        widget.setScaledContents(True)
    # 图片Base64
    if image_b64:
        widget.setPixmap(QPixmap(QImage.fromData(base64.b64decode(image_b64))))
        widget.setScaledContents(True)
    # 点击事件
    if clicked:
        widget.clicked.connect(clicked)
    return widget


def message(text: str, title: str = '提示'):
    """显示消息对话框，需要先初始化qApp.current_window
    """

    from daemon.component.dialog.message_dialog import MessageDialog
    return MessageDialog(text, title).exec_()


def confirm(text: str, title: str = '提示'):
    """显示消息对话框，需要先初始化qApp.current_window
    """

    from daemon.component.dialog.confirm_dialog import ConfirmDialog
    return ConfirmDialog(text, title).exec_()


def input_text(prompt: str = '', text: str = '', title: str = '提示'):
    """显示消息对话框，需要先初始化qApp.current_window
    """

    from daemon.component.dialog.input_dialog import InputDialog
    return InputDialog(prompt, text, title).exec_()


def directory_dialog() -> str or None:
    """显示文件夹选择对话框，默认打开桌面路径
    """

    try:
        return QFileDialog().getExistingDirectory(None, '选择文件夹', filesystem.desktop_dir)
    except Exception:
        return None


def file_dialog() -> str or None:
    """显示文件选择对话框，默认打开桌面路径，返回文件列表
    """

    try:
        selected_files, _ = QFileDialog().getOpenFileNames(None, '选择文件', filesystem.desktop_dir)
        return selected_files
    except Exception:
        return None


def loading(text: str):
    """加载中对话框
    """

    global loading_dialog
    if loading_dialog is None:
        from daemon.component.dialog.loading_dialog import LoadingDialog
        loading_dialog = LoadingDialog()
    loading_dialog.loading(text)


def hide_loading():
    """隐藏加载中对话框
    """

    if loading_dialog is not None:
        loading_dialog.stop()


def show_window(window: QWidget):
    """居中显示
    """

    if window.isMinimized():
        window.showNormal()
    if window.isVisible():
        window.activateWindow()
    else:
        window.show()
        move_center(window)
    return True


def move_center(window: QWidget):
    """移动居中
    """

    qr = window.frameGeometry()
    qr.moveCenter(QDesktopWidget().availableGeometry().center())
    window.move(qr.topLeft().x(), qr.topLeft().y())


def screen_size() -> QSize:
    """获取屏幕尺寸
    """

    global __screen_size
    if __screen_size is None:
        desktop = QApplication.desktop().availableGeometry()
        __screen_size = QSize(desktop.width(), desktop.height())
        # import Xlib.display
        # screen = Xlib.display.Display().screen()
        # __screen_size = QSize(screen.width_in_pixels, screen.height_in_pixels)
    return __screen_size


def install_translator(app: QApplication, lang_path: str):
    """设置系统语言为中文
    """

    app.translator = QTranslator()
    app.translator.load(lang_path)
    app.installTranslator(app.translator)


def is_maximize(window: QWidget):
    """判断是不是最大化
    """

    return window.width() == screen_size().width()


def scale_rate():
    """显示缩放系数
    """

    global __scale_rate
    if __scale_rate is None:
        __scale_rate = QApplication.screens()[0].logicalDotsPerInch() / 96
    return __scale_rate


def draw_shadow(painter: QPainter, width: int, height: int):
    """绘制控件阴影
    """

    shadow_width = 10
    radius = 8
    path = QPainterPath()
    path.setFillRule(Qt.WindingFill)
    path.addRoundedRect(shadow_width, shadow_width, width - radius * 2, height - shadow_width * 2, radius, radius)
    color = QColor(0, 0, 0)
    for i in range(shadow_width):
        path = QPainterPath()
        path.setFillRule(Qt.WindingFill)
        path.addRoundedRect(shadow_width - i, shadow_width - i, width - (shadow_width - i) * 2,
                            height - (shadow_width - i) * 2, radius + i, radius + i)
        color.setAlpha((shadow_width - i) * 1.2)
        painter.setPen(color)
        painter.drawPath(path)


def __word_wrap(widget: QWidget, text: str, text_line: int):
    """文字换行
    """

    fm = QFontMetrics(widget.font())
    line_list = []
    line_words = ''
    for word in text.split(' '):
        # 处理多个空字符串连在一起的情况
        if word == '':
            word = ' '
        # 拼接tmp_words
        if line_words == '':
            tmp_words = word
        else:
            tmp_words = line_words + ' ' + word

        # 判断tmp_words的宽度，如果小于控件宽度，则进入下一次循环继续拼接
        if fm.width(tmp_words) < widget.width():
            line_words = tmp_words

        # 如果大于控件宽度
        else:
            # 拼接上一次的状态
            if line_words != '':
                line_list.append(line_words)
            # 裁剪、拼接当前单词
            line_words = ''
            while len(word) > 0:
                line_words += word[0]
                word = word[1:]
                if fm.width(line_words) > widget.width() and '\u4e00' <= line_words[-1] <= '\u9fff':
                    line_list.append(line_words[:-1])
                    line_words = line_words[-1]

    # 拼接最后一个单词，可能会造成多一个空行，但下面的代码中会进行裁剪
    line_list.append(line_words)

    # 如果显示行数小于设置行数，补满空行
    if len(line_list) < text_line:
        while len(line_list) < text_line:
            line_list.append('')

    # 如果显示行数大于等于设置行数，拼接未显示的内容到最后一行，并进行裁剪
    else:
        last_line = ' '.join(line_list[text_line - 1:])
        if fm.width(last_line) > widget.width():
            while fm.width(last_line + '...') > widget.width() or last_line[-1] == ' ':
                last_line = last_line[:-1]
            line_list[text_line - 1] = last_line + '...'

    # 显示换行、裁剪后的内容，并重新设置高度
    widget.setText('\n'.join(line_list[:text_line]))
    widget.setFixedHeight((fm.height() + 4) * text_line)
