# -*- coding: utf-8 -*-
import platform
import subprocess


def cmd(command: str, cwd: str = None) -> list:
    """同步执行cmd命令，等待执行完毕后返回结果集合，如：[rs1, ...]
    Linux默认UTF-8编码，Windows默认GBK编码
    """

    output_lines = []
    ps = subprocess.Popen(command, cwd=cwd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    for line in ps.stdout.readlines():
        try:
            line_rows = str(line, encoding='GBK' if platform.system() == 'Windows' else 'UTF-8').strip()
        except UnicodeDecodeError:
            line_rows = str(line).strip()
        output_lines.append(line_rows)
    return output_lines


def cmd_async(command: str, cwd: str = None) -> subprocess.Popen:
    """异步执行cmd命令，不阻塞等待返回结果，直接返回ps对象
    """

    # 设置preexec_fn=os.setsid参数会导致运行第二次subprocess.Popen崩溃
    # 去除shell=True参数导致Linux下程序崩溃
    # 如需以新进程组启动软件，可用参数creationflags=subprocess.CREATE_NEW_PROCESS_GROUP
    ps = subprocess.Popen(command, cwd=cwd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    return ps


def pkexec(command: str, cwd: str = None):
    """pkexec同步执行cmd命令
    """

    output_lines = []
    ps = subprocess.Popen('pkexec sh -c "{}"'.format(command), cwd=cwd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    for line in ps.stdout.readlines():
        try:
            line_rows = str(line, encoding='GBK' if platform.system() == 'Windows' else 'UTF-8').strip()
        except UnicodeDecodeError:
            line_rows = str(line).strip()
        output_lines.append(line_rows)
    return output_lines
