# -*- coding: utf-8 -*-
import json
import time
import tqdm
import logging
import requests

# 网络请求session对象及创建时间，3分钟内不重新创建
__session = None
__session_t = 0.0
__SESSION_TIMEOUT_SECONDS = 60.0 * 3


def post(url: str, params: dict = None, files: dict = None, timeout: int = 30) -> dict:
    """POST请求，阻塞等待请求完成，返回响应字典或异常信息
    """

    request_time = time.time()
    resp = {'code': -1, 'msg': '', 'data': None}
    try:
        if params is None:
            params = {}

        text = __post(url, params, files, timeout).text
        data = json.loads(text)

        resp['code'] = data.get('code', -1)
        resp['msg'] = data.get('msg', '')
        resp['data'] = data.get('data', None)

    # 发生异常时，直接在msg字段返回异常信息
    except RuntimeError as e:
        resp['msg'] = str(e)
    except json.JSONDecodeError:
        resp['msg'] = f'返回数据解析错误 [{text}]'

    # 输出日志
    __print_log(url, params, resp, request_time)
    return resp


def post_bytes(url: str, params: dict, files: dict = None, timeout: int = 30) -> bytes:
    """POST请求，阻塞等待请求完成，返回bytes或抛出异常
    """

    data = __post(url, params, files, timeout).content
    return data


def wget(url: str, save_path: str):
    """下载文件，失败时抛出异常
    """

    response = requests.get(url, stream=True)
    if response.headers.get('Content-Length'):
        size = int(response.headers['Content-Length']) / 1024 / 1024
    else:
        size = 0
    with open(save_path, 'wb') as f:
        for data in tqdm.tqdm(iterable=response.iter_content(chunk_size=1024 * 1024), total=size, desc='正在下载', unit='MB'):
            f.write(data)


def __post(url: str, params: dict, files: dict = None, timeout: int = 30) -> requests.Response:
    """实际执行网络请求的方法，返回Response对象
    """

    global __session, __session_t
    now_t = time.time()
    if __session is None or now_t - __session_t > __SESSION_TIMEOUT_SECONDS:
        __session = requests.session()
        __session_t = now_t

    # 发起请求
    try:
        response = __session.post(url, json=params, files=files, timeout=timeout)

    # 处理异常
    except requests.exceptions.InvalidURL as e:
        raise RuntimeError(f'服务器IP错误：[{e}]')
    except requests.exceptions.ConnectionError as e:
        raise RuntimeError(f'无法连接到服务器：[{e}]')
    except requests.exceptions.Timeout as e:
        raise RuntimeError(f'连接服务器超时：[{e}]')
    except Exception as e:
        raise RuntimeError(f'网络请求出错：[{e}]')
    return response


def __print_log(url: str, params: dict, resp: dict, request_time: float):
    """输出请求日志
    """

    # 计算请求消耗时间
    cost_time = round(time.time() - request_time, 3)

    # 省略过长的参数
    for key, value in params.items():
        if len(str(value)) > 64:
            params[key] = str(value)[:63] + '...'

    if len(str(resp)) > 64:
        resp = str(resp)[:63] + '...'

    logging.warning(f'请求：{url}，参数：{params}，返回：{resp}，耗时：{cost_time}')
