# -*- coding: utf-8 -*-
import time
import logging
from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.common.by import By
from common import const, filesystem, api

# ChromeDriver
__driver = None
# Chrome操作锁
__chrome_lock = False


def change_goods(goods_index: int):
    """切换讲解商品
    """

    global __chrome_lock

    # 等待锁释放后加锁
    while __chrome_lock:
        time.sleep(0.2)
    __chrome_lock = True

    try:
        driver = __get_driver()

        # 检查网址
        if not driver.current_url.startswith('https://buyin.jinritemai.com/dashboard/live/control'):
            driver.execute_script('window.location.href = "https://buyin.jinritemai.com/dashboard/live/control";')
            time.sleep(2)

        # 判断巨量百应登陆状态
        if '抖音电商官网' in driver.title:
            api.add_alert(const.LEVEL_WARNING, '巨量百应账号未登录')
            return

        # 上下滚动页面，查找讲解商品
        scrollable_elements = __find_scrollable_elements(driver)
        scroll_count = 0
        while True:
            if scroll_count > 60:
                logging.warning('查找讲解商品超时')
                return

            goods_item_list = __find_invisible_elements(driver, By.CLASS_NAME, 'goodsItem-KBGOY5')
            if len(goods_item_list) == 0:
                logging.warning('查找讲解商品失败')
                return

            min_number = int(goods_item_list[0].find_elements(By.CLASS_NAME, 'auxo-input')[0].get_attribute('value'))
            max_number = int(goods_item_list[-1].find_elements(By.CLASS_NAME, 'auxo-input')[0].get_attribute('value'))

            if min_number > goods_index:
                for element in scrollable_elements:
                    driver.execute_script('arguments[0].scrollTop -= 1000;', element)
                scroll_count += 1
                time.sleep(0.5)
                continue

            elif max_number < goods_index:
                for element in scrollable_elements:
                    driver.execute_script('arguments[0].scrollTop += 1000;', element)
                scroll_count += 1
                time.sleep(0.5)
                continue

            else:
                logging.info('在当前页面发现讲解商品')
                break

        # 查找要点击的商品
        goods_item_list = driver.find_elements(By.CLASS_NAME, 'goodsItem-KBGOY5')
        for goods_item in goods_item_list:
            number_input = goods_item.find_elements(By.CLASS_NAME, 'auxo-input')[0]
            number = int(number_input.get_attribute('value'))

            # 找到商品后，只在开播后点击讲解，未开播时不点击下架
            if number == goods_index:
                jiangjie_button = goods_item.find_elements(By.TAG_NAME, f'button')[0]
                if jiangjie_button.text == '讲解':
                    driver.execute_script('arguments[0].click()', jiangjie_button)

    except Exception as e:
        logging.warning(f'切换讲解商品失败：{e}')

    # 释放锁
    __chrome_lock = False


def send_message(message: dict):
    """发送弹幕
    """

    global __chrome_lock

    # 等待锁释放后加锁
    while __chrome_lock:
        time.sleep(0.2)
    __chrome_lock = True


    try:
        driver = __get_driver()

        # 刷新网址
        if not driver.current_url.startswith('https://buyin.jinritemai.com/dashboard/live/control?'):
            driver.execute_script('window.location.href = "https://buyin.jinritemai.com/dashboard/live/control?";')
            time.sleep(20)

        # 判断巨量百应登陆状态
        if '抖音电商官网' in driver.title:
            api.add_alert(const.LEVEL_WARNING, '巨量百应账号未登录')
            return

        # 找到输入框
        elements = __find_invisible_elements(driver, By.XPATH, '//*[@id="input-comment-block-id"]/div[1]/span/textarea')
        if len(elements) == 1:
            elements[0].send_keys(f'@{message["user"]}，{message["content"]}')

        # 点击发送
        elements = __find_invisible_elements(driver, By.CLASS_NAME, 'inputSuffix-qJWii2 ')
        if len(elements) == 1:
            elements[0].click()

    except Exception as e:
        logging.warning(f'发送弹幕失败：{e}')

    # 释放锁
    __chrome_lock = False


def get_trade_info() -> dict:
    """抓取经营数据，判断巨量百应登陆状态
    {douyin_name: str, today_total_amount: float, today_total_count: int, today_income_amount: float, live_hours: float, viewer_count: float}
    """

    global __chrome_lock

    # 等待锁释放后加锁
    while __chrome_lock:
        time.sleep(0.2)
    __chrome_lock = True

    trade_info = {'douyin_name': None,
                  'today_total_amount': None,
                  'today_total_count': None,
                  'today_income_amount': None,
                  'live_hours': None,
                  'viewer_count': None}
    try:
        driver = __get_driver()

        # 刷新网址
        if not driver.current_url.startswith('https://buyin.jinritemai.com/dashboard?'):
            driver.execute_script('window.location.href = "https://buyin.jinritemai.com/dashboard?";')
            time.sleep(20)

        # 判断巨量百应登陆状态
        if '抖音电商官网' in driver.title:
            api.add_alert(const.LEVEL_WARNING, '巨量百应账号未登录')
            return trade_info

        elements = __find_invisible_elements(driver, By.CLASS_NAME, 'index_module__dataTypeNum___b535e')
        # 当日成交金额
        trade_info['today_total_amount'] = float(elements[0].text.replace('¥', '').replace('￥', '').replace(',', ''))

        # 当日成交订单量
        trade_info['today_total_count'] = int(elements[1].text.replace('¥', '').replace('￥', '').replace(',', ''))

        # 当日佣金收入
        trade_info['today_income_amount'] = float(elements[3].text[1:].replace('¥', '').replace('￥', '').replace(',', ''))

        elements = driver.find_elements(By.CLASS_NAME, 'index_module__contentItemNum____9088')
        if len(elements) == 2:
            # 本场开播时长
            live_time = elements[0].text.split(':')
            live_time_h = int(live_time[0])
            live_time_m = int(int(live_time[1]) / 60 * 10) + 1
            if live_time_m == 10:
                live_time_h += 1
                live_time_m = 0
            trade_info['live_hours'] = float(f'{live_time_h}.{live_time_m}')

            # 实时看播人数
            trade_info['viewer_count'] = int(elements[1].text.replace(',', ''))

        elements = driver.find_elements(By.CLASS_NAME, 'btn-item-role-exchange-name__title')
        if len(elements) == 1:
            trade_info['douyin_name'] = elements[0].text

    except Exception as e:
        logging.warning(f'抓取经营数据失败：{e}')

    # 释放锁
    __chrome_lock = False

    return trade_info


def __get_driver() -> webdriver.Chrome:
    """获取ChromeDriver，可能会抛出异常
    """

    global __driver
    # 连接Chrome端口，Chrome未打开时会阻塞
    options = webdriver.ChromeOptions()
    options.add_experimental_option('debuggerAddress', f'127.0.0.1:{const.CHROME_PORT}')
    __driver = webdriver.Chrome(service=Service(filesystem.chrome_driver_path), options=options)
    return __driver


def __find_invisible_elements(driver: webdriver.Chrome, by: str, value: str) -> list:
    """查找未渲染的元素，先最大化，查找结束后最小化
    """

    try:
        driver.maximize_window()
    except Exception:
        pass
    time.sleep(0.5)
    element_list = driver.find_elements(by, value)
    try:
        driver.minimize_window()
    except Exception:
        pass
    return element_list


def __find_scrollable_elements(driver: webdriver.Chrome) -> list:
    """查找可滚动的组件
    """

    scrollable_elements = driver.execute_script("""
      var scrollableElements = [];
      var allElements = document.getElementsByTagName('*');
      for (var i = 0; i < allElements.length; i++) {
        var element = allElements[i];
        var computedStyle = window.getComputedStyle(element);
        if ((computedStyle.overflowY === 'auto' || computedStyle.overflowY === 'scroll') &&
            element.clientHeight < element.scrollHeight) {
          scrollableElements.push(element);
        }
      }
      return scrollableElements;
    """)
    return scrollable_elements
