# -*- coding: utf-8 -*-
import time
import json
import logging
import traceback
from abc import ABC
from tornado.web import RequestHandler


class BaseHandler(RequestHandler, ABC):

    def prepare(self):
        """入口函数
        """

        # 记录请求时间
        self.request_time = time.time()

        # 记录客户端IP
        self.client_ip = self.request.remote_ip

        # 解析请求参数
        self.params = self.__parse_params()

    def set_default_headers(self):
        """设置默认响应头
        """

        self.add_header('Access-Control-Allow-Origin', '*')
        self.add_header('Access-Control-Allow-Methods', 'POST, GET, OPTIONS, PUT')
        # axios浏览器跨域预检处理
        self.add_header('Access-Control-Allow-Headers', 'Content-Type')

    def write_error(self, status_code: int, **kwargs: any):
        """全局异常处理
        """

        excp = kwargs['exc_info'][1]
        traceback_list = traceback.format_exception(*kwargs['exc_info'])
        resp = {'code': -1, 'msg': str(excp), 'data': '\n'.join(traceback_list)}

        # 返回异常信息
        self.set_status(200)
        self.write(resp)

        # 输出错误日志
        self.__print_log(resp, is_error_log=True)

    def options(self):
        """axios浏览器跨域预检处理
        """

        self.set_status(204)
        self.finish()

    async def post(self):
        """POST请求
        """

        # 处理业务
        resp = await self.handle_request()
        self.write(resp)

        # 输出日志
        self.__print_log(resp)

    async def handle_request(self) -> dict:
        """处理POST请求的具体方法，先检查登陆是否过期
        """

        pass

    def arg(self, key: str):
        """获取请求参数
        """

        return self.params[key]

    def argf(self, key: str):
        """从请求的files表单中获取文件
        """

        return self.request.files[key][0]

    def __print_log(self, resp: any, is_error_log: bool = False):
        """输出日志
        """

        return

        cost_time = round(time.time() - self.request_time, 3)
        # 省略过长的参数
        for key, value in self.params.items():
            if len(str(value)) > 64:
                self.params[key] = str(value)[:63] + '...'

        if len(str(resp)) > 64:
            resp = str(resp)[:63] + '...'

        state = '成功' if not is_error_log else '失败'
        logging.info(f'用户IP:{self.client_ip}，请求{self.request.full_url()}{state}，耗时:{cost_time}，参数:{self.params}，返回:{resp}')

    def __parse_params(self) -> dict:
        """解析请求参数
        """

        params = {}
        if self.request.body != b'':
            try:
                params = json.loads(self.request.body)
            except Exception:
                pass
                # logging.info(f'请求参数解析出错：{self.request.body}')
        return params
