# -*- coding: utf-8 -*-
from PyQt5.Qt import Qt, QWidget, QMouseEvent, QColor, QPaintEvent, QPainter, QPixmap, QPainterPath, QPen, \
    QPropertyAnimation, QIcon, QGraphicsDropShadowEffect
from daemon.component.title_bar import TitleBar
from daemon.component.layout import VBoxLayout
from daemon.helper import ui_helper
from common import filesystem


class BaseWindowUI(QWidget):

    def __init__(self, title: str, parent: QWidget = None, resizeable: bool = False):
        """窗口界面
        """

        super(BaseWindowUI, self).__init__(parent)
        self.setWindowTitle(title)
        self.setWindowIcon(QIcon(filesystem.ic_logo))

        self.title = title
        self.resizeable = resizeable

        # 鼠标按下时的坐标
        self.__mouse_press_pos = None

        # 无边框透明，注意：KylinV10不能设置WindowMinimizeButtonHint否则无边框会失效
        self.setWindowFlags(Qt.FramelessWindowHint | Qt.WindowCloseButtonHint | Qt.WindowMinimizeButtonHint)
        self.setAttribute(Qt.WA_TranslucentBackground, True)

        # 阴影
        shadow = QGraphicsDropShadowEffect()
        shadow.setOffset(0, 0)
        shadow.setColor(QColor('#999999'))
        shadow.setBlurRadius(16)
        self.setGraphicsEffect(shadow)

        # 显示和关闭动画
        self.show_anim = None
        self.close_anim = None

        # 窗口布局
        self.title_bar = self.init_title_bar()
        self.title_bar.maximize_signal.connect(self.__handle_maximize)
        self.body_widget = self.init_body()
        self.body_width = self.body_widget.width()
        self.body_height = self.body_widget.height()

        self.main_layout = VBoxLayout()
        self.main_layout.addWidget(self.title_bar)
        self.main_layout.addSpacing(8)
        self.main_layout.addWidget(self.body_widget)
        self.setLayout(self.main_layout)

        # 设置边距，相对定位的body_widget通过padding与标题栏对齐
        self.main_layout.setContentsMargins(26, 10, 20, 10)
        self.body_widget.setContentsMargins(0, 0, 6, 16)

    def init_body(self) -> QWidget:
        """子类必须重写，初始化窗口主体
        """

        pass

    def init_title_bar(self) -> TitleBar:
        """子类可以重写，设置自定义标题栏
        """

        return TitleBar(self.title, resizeable=self.resizeable)

    def paintEvent(self, event: QPaintEvent):
        """绘制界面
        """

        # 抗锯齿画笔
        painter = QPainter(self)
        painter.setRenderHint(QPainter.Antialiasing, True)
        painter.setPen(QPen(QColor(200, 200, 200), 1))
        bg_pixmap = QPixmap(self.width(), self.height())
        bg_pixmap.fill(QColor(255, 255, 255))
        bg_path = QPainterPath()

        # 非最大化
        if not ui_helper.is_maximize(self):
            painter.drawRoundedRect(10, 10, self.width() - 20, self.height() - 20, 8, 8)
            bg_path.addRoundedRect(10, 10, self.width() - 20, self.height() - 20, 8, 8)

        # 最大化
        else:
            painter.drawRect(0, 0, self.width(), self.height())
            bg_path.addRect(0, 0, self.width(), self.height())

        painter.setClipPath(bg_path)
        painter.drawPixmap(0, 0, self.width(), self.height(), bg_pixmap)

    def mousePressEvent(self, event: QMouseEvent):
        """鼠标点击事件
        """

        if event.buttons() == Qt.LeftButton and self.title_bar.geometry().contains(event.pos()):
            self.__mouse_press_pos = event.globalPos() - self.pos()

    def mouseMoveEvent(self, event: QMouseEvent):
        """鼠标移动事件
        """

        if self.__mouse_press_pos is not None:
            # 最大化时拖拽标题栏，先还原窗口再移动
            if ui_helper.is_maximize(self):
                self.__normal()
                self.__mouse_press_pos.setX(self.__mouse_press_pos.x() * (self.width() / ui_helper.screen_size().width()))

            self.setCursor(Qt.SizeAllCursor)
            self.move(event.globalPos() - self.__mouse_press_pos)

    def mouseReleaseEvent(self, event: QMouseEvent):
        """鼠标释放事件
        """

        self.__mouse_press_pos = None
        self.setCursor(Qt.ArrowCursor)

    def show(self):
        """重写显示事件，增加动画效果，同时激活窗口
        """

        if self.show_anim is None:
            self.__normal()
            super(BaseWindowUI, self).show()
            self.show_anim = self.__anim(0, 1)
            self.show_anim.start()

    def close(self):
        """重写关闭事件，增加动画效果
        """

        if self.close_anim is None:
            self.close_anim = self.__anim(1, 0)
            self.close_anim.start()
            self.close_anim.finished.connect(self.__handle_close_animation_finish)

    def __handle_close_animation_finish(self):
        """窗口关闭动画结束后调用真正的关闭方法
        """

        super(BaseWindowUI, self).close()
        self.show_anim = None
        self.close_anim = None

    def __anim(self, start_value: int, end_value: int):
        """淡出淡入动画效果
        """

        anim = QPropertyAnimation(self, b'windowOpacity')
        anim.setDuration(150)
        anim.setStartValue(start_value)
        anim.setEndValue(end_value)
        return anim

    def __handle_maximize(self):
        """窗口最大化、还原
        """

        if ui_helper.is_maximize(self):
            self.__normal()
        else:
            self.__maximize()

    def __normal(self):
        """还原：title_bar + body_widget + spacing(8) + padding(26, 10, 20, 10)
        """

        self.setFixedSize(self.body_width + 46, self.body_height + self.title_bar.height() + 28)
        self.body_widget.setFixedSize(self.body_width, self.body_height)
        ui_helper.move_center(self)
        self.main_layout.setContentsMargins(26, 10, 20, 10)

    def __maximize(self):
        """最大化：title_bar + body_widget + spacing(8) + padding(16, 0, 10, 0)
        """

        self.move(0, 0)
        self.setFixedSize(ui_helper.screen_size())
        self.body_widget.setFixedSize(self.width() - 26, self.height() - self.title_bar.height() - 8)
        self.main_layout.setContentsMargins(16, 0, 10, 0)
