# -*- coding: utf-8 -*-
from PyQt5.Qt import Qt, QWidget, QVBoxLayout, QDialog, QMouseEvent, QPaintEvent, QPainter, QLabel, QColor,\
    QPainterPath, QPixmap, QEvent, QTimer, QKeyEvent, QGraphicsDropShadowEffect, pyqtSignal
from daemon.helper import ui_helper


class ContextMenu(QDialog):

    def __init__(self, parent: QWidget = None, width: int = 220, drop_down_mode: bool = False, close_callback: object = None):
        """右键菜单
        drop_down_mode: 是否为下拉菜单
        """

        super(ContextMenu, self).__init__(parent)
        self.setFixedWidth(width + 20)
        self.drop_down_mode = drop_down_mode
        self.close_callback = close_callback

        # 无边框透明
        self.setWindowFlags(Qt.FramelessWindowHint | Qt.Popup | Qt.NoDropShadowWindowHint)
        self.setAttribute(Qt.WA_TranslucentBackground, True)

        # 阴影
        shadow = QGraphicsDropShadowEffect()
        shadow.setOffset(0, 0)
        shadow.setColor(QColor('#666666'))
        shadow.setBlurRadius(8)
        self.setGraphicsEffect(shadow)

        # 淡出淡入
        if not drop_down_mode:
            self.setWindowOpacity(0)
            self.opacity_timer = QTimer(self)
            self.opacity_timer.timeout.connect(self.__handle_opacity_timer_timeout)
            self.opacity_timer.start(20)

        # 按钮布局
        self.main_v_layout = QVBoxLayout()
        self.main_v_layout.setSpacing(0)
        self.setLayout(self.main_v_layout)

    def paintEvent(self, event: QPaintEvent):
        """绘制界面背景
        """

        # 创建画笔
        painter = QPainter(self)
        # 抗锯齿
        painter.setRenderHint(QPainter.Antialiasing, True)

        # 圆角矩形边框
        bg_round_rect_path = QPainterPath()
        bg_round_rect_path.addRoundedRect(10, 10, self.width() - 20, self.height() - 20, 2, 2)
        painter.setClipPath(bg_round_rect_path)

        # 背景
        bg_pixmap = QPixmap(self.width(), self.height())
        bg_pixmap.fill(QColor(255, 255, 255))
        painter.drawPixmap(10, 10, self.width() - 20, self.height() - 20, bg_pixmap)

    def mousePressEvent(self, event: QMouseEvent):
        """鼠标点击事件，点击任意位置关闭
        """

        if self.close_callback:
            self.close_callback()
        else:
            self.close()

    def keyPressEvent(self, event: QKeyEvent):
        """重写ESC行为
        """

        if event.key() == Qt.Key_Escape and self.auto_close:
            if self.close_callback:
                self.close_callback()
            else:
                self.close()

    def add_action(self, text: str, on_click: callable):
        """添加菜单选项
        """

        self.main_v_layout.addWidget(_MenuAction(text, self, on_click))

    def __handle_opacity_timer_timeout(self):
        """淡出淡入设置透明度
        """

        if self.windowOpacity() < 0.9:
            self.setWindowOpacity(self.windowOpacity() + 0.05)
        else:
            self.opacity_timer.stop()


class _MenuAction(QWidget):

    # 点击事件信号
    clicked = pyqtSignal()

    def __init__(self, text: str, parent: QWidget = None, on_clicked: callable = None):
        """菜单按钮，宽66高31
        """

        super(_MenuAction, self).__init__(parent)
        self.setFixedSize(parent.width() - 22, 31)
        self.is_active = False

        # 文字标签
        self.label = QLabel(self)
        left_padding = 10 if parent.drop_down_mode else 30

        ui_helper.attr(self.label, text=text, width=self.width() - left_padding, height=self.height(),
                       color='#333333', font_size=10, align=Qt.AlignLeft | Qt.AlignVCenter,
                       other_style='padding-left: {}px;text-overflow: ellipsis;'.format(left_padding))

        # 鼠标形状
        self.setCursor(Qt.PointingHandCursor)

        # 点击事件
        if on_clicked:
            self.clicked.connect(on_clicked)

    def paintEvent(self, event: QPaintEvent):
        """绘制背景
        """

        # 创建画笔
        painter = QPainter(self)
        # 绘制边框
        border_color = QColor(255, 255, 255, 0)
        painter.setPen(border_color)
        painter.drawRect(0, 0, self.width(), self.height())
        bg_round_rect_path = QPainterPath()
        bg_round_rect_path.addRect(0, 0, self.width(), self.height())
        painter.setClipPath(bg_round_rect_path)
        # 绘制背景
        bg_color = QColor(230, 230, 230, 128) if self.is_active else QColor(255, 255, 255)
        bg_pixmap = QPixmap(self.width(), self.height())
        bg_pixmap.fill(bg_color)
        painter.drawPixmap(0, 0, self.width(), self.height(), bg_pixmap)

    def mousePressEvent(self, event: QMouseEvent):
        """鼠标按下事件
        """

        self.label.move(self.label.x() + 1, self.label.y() + 1)

    def mouseReleaseEvent(self, event: QMouseEvent):
        """鼠标释放事件
        """

        self.label.move(self.label.x() - 1, self.label.y() - 1)
        if event.button() == Qt.LeftButton and self.geometry().contains(event.pos() + self.pos()):
            if self.parent().close_callback:
                self.parent().close_callback()
            else:
                self.parent().close()
            self.clicked.emit()

    def enterEvent(self, event: QEvent):
        """鼠标进入事件
        """

        self.is_active = True
        self.repaint()

    def leaveEvent(self, event: QEvent):
        """鼠标离开事件
        """

        self.is_active = False
        self.repaint()
