# -*- coding: utf-8 -*-
from PyQt5.Qt import Qt, QMouseEvent, QPixmap, QLabel, QImage, QWidget, QEvent, pyqtSignal
from daemon.component.layout import HBoxLayout
from daemon.helper import ui_helper
from common import filesystem


class TitleBar(QWidget):

    maximize_signal = pyqtSignal()

    def __init__(self, title: str, parent: QWidget = None, resizeable: bool = False, minimizable: bool = True):
        """窗口标题栏
        """

        super(TitleBar, self).__init__(parent)
        self.setFixedHeight(45)
        self.resizeable = resizeable

        # 标题
        self.icon_label = QLabel()
        ui_helper.attr(self.icon_label, width=22, height=22, image_path=filesystem.ic_title_logo)
        self.title_label = QLabel(title)
        ui_helper.attr(self.title_label, font_size=13)
        self.title_label.setContentsMargins(0, 0, 0, 2)

        # 最小化
        minimize_button = _TitleActionButton(filesystem.ic_minimize, filesystem.ic_minimize_active)
        minimize_button.clicked.connect(lambda: self.parentWidget().showMinimized())

        # 最大化
        maximize_button = _TitleActionButton(filesystem.ic_maximize, filesystem.ic_maximize_active)
        maximize_button.clicked.connect(lambda: self.maximize_signal.emit())

        # 关闭
        self.close_button = _TitleActionButton(filesystem.ic_close, filesystem.ic_close_active)
        self.close_button.clicked.connect(lambda: self.parentWidget().close())

        # 设置布局
        bar_h_layout = HBoxLayout()
        bar_h_layout.addWidget(self.icon_label)
        bar_h_layout.addSpacing(6)
        bar_h_layout.addWidget(self.title_label)
        bar_h_layout.addStretch(1)
        if minimizable:
            bar_h_layout.addWidget(minimize_button)
        if resizeable:
            bar_h_layout.addWidget(maximize_button)
        bar_h_layout.addWidget(self.close_button)
        self.setLayout(bar_h_layout)

    def mouseDoubleClickEvent(self, event: QMouseEvent):
        """双击事件
        """

        if self.resizeable and event.button() == Qt.LeftButton:
            self.maximize_signal.emit()


class _TitleActionButton(QWidget):

    clicked = pyqtSignal()

    def __init__(self, icon: str, active_icon: str, parent: QWidget = None):
        """标题栏按钮 40x40
        """

        super(_TitleActionButton, self).__init__(parent)
        self.setFixedSize(40, 40)
        self.icon = icon
        self.active_icon = active_icon
        self.is_active = False
        self.label = QLabel(self)
        self.label.setFixedSize(40, 40)
        self.label.setPixmap(QPixmap.fromImage(QImage(icon)))
        self.label.setScaledContents(True)
        self.label.setContentsMargins(10, 10, 10, 10)
        self.label.setMouseTracking(True)
        self.setCursor(Qt.PointingHandCursor)

    def enterEvent(self, event: QEvent):
        """鼠标进入事件
        """

        self.label.setPixmap(QPixmap.fromImage(QImage(self.active_icon)))

    def leaveEvent(self, event: QEvent):
        """鼠标离开事件
        """

        self.label.setPixmap(QPixmap.fromImage(QImage(self.icon)))

    def mousePressEvent(self, event: QMouseEvent):
        """鼠标点击事件
        """

        event.accept()

    def mouseReleaseEvent(self, event: QMouseEvent):
        """鼠标释放事件
        """

        if event.button() == Qt.LeftButton and self.geometry().contains(event.pos() + self.pos()):
            self.clicked.emit()
