# -*- coding: utf-8 -*-
import os
import time
import psutil
import shutil
import base64
import socket
import getmac
import platform
import datetime
import threading
from io import BytesIO
from PIL import Image

# 全局列表持有子线程，避免线程意外退出
__gthread_list = []


def now_time(fmt: str = '%Y-%m-%d %H:%M:%S') -> str:
    """获取当前时间
    """

    return datetime.datetime.now().strftime(fmt)


def timestamp(fmt_time: str = None, fmt: str = '%Y-%m-%d %H:%M:%S') -> int:
    """当前时间转换为时间戳
    """

    if fmt_time is None:
        return int(time.time())
    return int(time.mktime(time.strptime(fmt_time, fmt)))


def from_timestamp(_timestamp: int, fmt: str = '%Y-%m-%d %H:%M:%S') -> str:
    """unix时间戳转时间字符串
    """

    return datetime.datetime.fromtimestamp(_timestamp).strftime(fmt)


def from_time(t: float, fmt: str = '%H:%M:%S'):
    """时间float转字符串
    """

    return time.strftime(fmt, time.localtime(t))


def is_win() -> bool:
    """判断本机是Windows
    """

    return platform.system() == 'Windows'


def get_host_ip(server_ip: str) -> str:
    """通过与服务器建立连接，获取本机IP，获取失败（如服务器连接失败）时抛出异常
    """

    sock = socket.socket()
    sock.connect((server_ip.split(':')[0], int(server_ip.split(':')[1])))
    return sock.getsockname()[0]


def encode_b64(text: str) -> bytes:
    """base64编码（不带等号！！！）
    """

    return base64.b64encode(bytes(text, 'UTF-8')).rstrip(b'=')


def decode_b64(b64text: bytes) -> str:
    """base64解码（不带等号！！！）
    """

    b64text += b'=' * (-len(b64text) % 4)
    return str(base64.b64decode(b64text), 'UTF-8')


def int_to_bytes(number: int) -> bytes:
    """数字转bytes
    """

    return bytes(str(number), 'UTF-8')


def bytes_to_int(b_number: bytes) -> int:
    """bytes转数字
    """

    return int(str(b_number, 'UTF-8'))


def read_file(path: str, mode: str = 'r', encoding: str or None = 'UTF-8') -> str:
    """读取文件，抛出异常（如文件不存在或权限不足）时返回空字符串
    """

    content = ''
    try:
        file = open(path, mode, encoding=encoding)
        content = file.read()
        file.close()
    except Exception:
        pass
    return content


def write_file(path: str, content: str or bytes = None, mode: str = 'w', encoding: str or None = 'UTF-8', chmod: int = 0o0644):
    """创建并写入文件，文件已存在时会覆盖原文件，抛出异常（如权限不足）时不会创建
    """

    try:
        # 如果文件夹不存在则先创建文件夹
        path_dir, _ = os.path.split(path)
        if not os.path.exists(path_dir):
            os.makedirs(path_dir)
        file = open(path, mode, encoding=encoding)
        # 写入文件内容
        if content:
            file.write(content)
        file.close()
        # 设置文件权限
        os.chmod(path, chmod)
    except Exception:
        pass


def get_mac(server_ip: str) -> str or None:
    """获取MAC地址
    """

    try:
        socket.setdefaulttimeout(2)
        socket.socket(socket.AF_INET, socket.SOCK_STREAM).connect((server_ip.split(':')[0], int(server_ip.split(':')[1])))
        return getmac.get_mac_address()
    except socket.error:
        return None
    finally:
        socket.setdefaulttimeout(None)


def rmfile(file_path: str):
    """删除文件，删除失败时不抛出异常
    """

    if os.path.exists(file_path):
        try:
            os.remove(file_path)
        except Exception:
            pass


def rmtree(dir_path: str):
    """删除文件夹中的所有文件，不删除文件夹中的文件夹
    """

    if os.path.isdir(dir_path):
        for file_name in os.listdir(dir_path):
            rmfile(os.path.join(dir_path, file_name))


def mkdir(dir_path: str):
    """创建文件夹
    """

    if not os.path.exists(dir_path):
        os.makedirs(dir_path)


def copyfile(src_path: str, dst_path: str) -> bool:
    """复制文件
    """

    try:
        shutil.copy(src_path, dst_path)
        return True
    except Exception:
        return False


def movefile(src_path: str, dst_path: str) -> bool:
    """移动文件
    """

    try:
        shutil.move(src_path, dst_path)
        return True
    except Exception:
        return False


def start_thread(target: callable, args: iter = ()):
    """启动线程
    """

    __gthread_list.append(threading.Thread(target=target, args=args))
    __gthread_list[-1].start()


def get_str_diff_count(str1: str, str2: str) -> int:
    """获取两个字符串的差异数量
    """

    # 计算两个字符串的长度
    len1, len2 = len(str1), len(str2)
    # 初始化一个二维数组用于DP
    dp = [[0] * (len2 + 1) for _ in range(len1 + 1)]

    for i in range(len1 + 1):
        dp[i][0] = i
    for j in range(len2 + 1):
        dp[0][j] = j

    # 动态规划计算编辑距离
    for i in range(1, len1 + 1):
        for j in range(1, len2 + 1):
            if str1[i - 1] == str2[j - 1]:
                dp[i][j] = dp[i - 1][j - 1]
            else:
                dp[i][j] = 1 + min(dp[i - 1][j], dp[i][j - 1], dp[i - 1][j - 1])

    # 如果编辑距离小于或等于diff_count，则字符串差异小于等于diff_count个字符
    diff_count = dp[len1][len2]
    return diff_count


def calc_duration_hours(calc_t: float) -> float:
    """计算经过的小时，保留1位小数
    """

    if int(calc_t) == 0:
        return 0

    duration_hours_str = str((time.time() - calc_t) / 3600)
    if '.' in duration_hours_str:
        duration_hours_str_split = duration_hours_str.split('.')
        duration_hours_float = duration_hours_str_split[1][:1]
        duration_hours_str = duration_hours_str_split[0] + '.' + duration_hours_float
    duration_hours = float(duration_hours_str) + 0.1
    return round(duration_hours, 1)


def get_process_start_timestamp(process_name) -> int:
    """获取进程启动时间戳
    """

    for proc in psutil.process_iter(['pid', 'name', 'create_time']):
        if process_name.lower() in proc.info['name'].lower():
            return proc.info['create_time']
    return 0


def get_image_b64(image: Image) -> str or None:
    """获取图片b64
    """

    image_buffer = BytesIO()
    try:
        image.save(image_buffer, format='JPEG')
        image_b64 = base64.b64encode(image_buffer.getvalue()).decode()
        return image_b64
    except Exception as e:
        return None


def snowid() -> int:
    """获取一个雪花ID
    """

    return __snowid.next_id()


class Snowid(object):
    """推特雪花ID（12位数字）
    https://github.com/twitter-archive/snowflake/blob/snowflake-2010/src/main/scala/com/twitter/service/snowflake/IdWorker.scala
    """

    # Twitter元年时间戳
    TWEPOCH = 1288834974657

    def __init__(self):
        """初始化
        """

        # 上次计算的时间戳
        self.last_timestamp = -1

    def next_id(self) -> int:
        """获取下一个雪花ID
        """

        timestamp = int(time.time() * 1000)
        if timestamp == self.last_timestamp:
            # 等到下一毫秒
            timestamp = int(time.time() * 1000)
            while timestamp <= self.last_timestamp:
                timestamp = int(time.time() * 1000)
        self.last_timestamp = timestamp
        next_id = timestamp - self.TWEPOCH
        return next_id


__snowid = Snowid()
