# -*- coding: utf-8 -*-
import wave
import logging


def get_wav_duration(wav_path: str) -> float:
    """获取音频时长（秒） TODO: 处理可能抛出的异常
    """

    with wave.open(wav_path, 'r') as wav_file:
        nframes = wav_file.getnframes()
        framerate = wav_file.getframerate()
        duration = nframes / float(framerate)
    return duration


def cropping_wav_blank(input_path: str, output_path: str, cropping_seconds: float = 0.2):
    """裁剪音频尾部空白
    """

    from common import filesystem
    from util import cmd_util
    duration = get_wav_duration(input_path)
    end_time = duration - cropping_seconds
    cmd_util.cmd(f'{filesystem.ffmpeg_path} -i "{input_path}" -to {end_time:.3f} -c copy "{output_path}"')


def concat_blank_wav(input_path: str, output_path: str, blank_count: int):
    """在音频尾部拼接空白，每个加号拼接一段500ms空白音频
    """

    from common import filesystem
    wav_list = [input_path]
    for _ in range(blank_count):
        wav_list.append(filesystem.blank_wav_path)
    concat_wav_list(wav_list, output_path)


def concat_wav_list(wav_list: list, output_path: str):
    """拼接音频 ffmpeg -i 1.wav -i 2.wav -i 3.wav -filter_complex '[0:0] [1:0] [2:0] concat=n=3:v=0:a=1 [a]' -map [a] output.wav
    """

    from common import filesystem
    from util import cmd_util
    input_list_str = ''
    array_list_str = ''
    for index, wav_path in enumerate(wav_list):
        input_list_str += f' -i "{wav_path}"'
        array_list_str += f'[{index}:0] '

    cmd = f'{filesystem.ffmpeg_path} {input_list_str} -filter_complex "{array_list_str} ' \
          f'concat=n={len(wav_list)}:v=0:a=1 [a]" -map [a] {output_path} -y -v quiet'
    logging.debug(f'合成音频：{cmd}')
    cmd_util.cmd(cmd)


def cut_wav(input_path: str, cut1_duration: float, output1_path: str, output2_path: str):
    """切分wav，第一段的时长为cut_duration，第二段的时长为剩余部分
    """

    with wave.open(input_path, 'rb') as wf:
        params = wf.getparams()
        nchannels, sampwidth, framerate, nframes = params[:4]
        duration = nframes / float(framerate)

        # 计算第一段切分的帧数
        frames_1 = wf.readframes(int(framerate * cut1_duration))
        with wave.open(output1_path, 'wb') as wf1:
            wf1.setparams(params)
            wf1.writeframes(frames_1)

        # 计算第二段切分的帧数
        frames_2 = wf.readframes(int(framerate * (duration - cut1_duration)))
        with wave.open(output2_path, 'wb') as wf2:
            wf2.setparams(params)
            wf2.writeframes(frames_2)
