# -*- coding: utf-8 -*-
import os
import shutil
import string
import logging
from common import filesystem, const
from util import common_util, request_util


def tts(text_list: list, is_assistant: bool) -> tuple:
    """调用TTS文本转语音，返回(wav_path_list, pure_text_list, goods_index_list)
    """

    # 转语音时去除文案尾部的加号，并记录加号的数量和商品序号
    pure_text_list, blank_count_list, goods_index_list = __exec_text_list(text_list)

    # 获取并保存wav
    wav_path_list = []
    if len(pure_text_list) > 0:
        wav_path_list = __tts(pure_text_list, blank_count_list, is_assistant)

    return wav_path_list, pure_text_list, goods_index_list


def __exec_text_list(text_list: list) -> tuple:
    """预处理文本列表，转语音时去除文案尾部的加号，并记录加号的数量和商品序号
    """

    pure_text_list = []
    blank_count_list = []
    goods_index_list = []
    for text in text_list:
        text = text.strip()

        # 获取商品序号，如：1#我们的香米来自大山梯田
        goods_index = None
        if '#' in text and text.split('#')[0].isdigit():
            goods_index = int(text.split('#')[0])
            text = text.split('#')[1]

        # 计算尾部加号数量
        blank_count = 0
        while text.endswith('+'):
            text = text[:-1]
            blank_count += 1

        if text != '':
            pure_text_list.append(text)
            blank_count_list.append(blank_count)
            goods_index_list.append(goods_index)

    return pure_text_list, blank_count_list, goods_index_list


def __tts(text_list: list, blank_count_list: list, is_assistant: bool, retry_count: int = 0) -> list:
    """执行TTS生成语音，返回wav文件路径列表
    """

    try:
        tts_wav_list = get_wav_list(text_list, is_assistant)

    except Exception as e:
        retry_count += 1
        logging.error(f'TTS生成失败，重试第{retry_count}次：{e}')
        return __tts(text_list, blank_count_list, is_assistant, retry_count)

    # 处理wav语音文件
    wav_path_list = []
    for index, tts_wav_path in enumerate(tts_wav_list):
        wav_id = common_util.snowid()

        # # 裁剪音频尾部空白
        # tmp_wav_path = os.path.join(filesystem.runtime_wav_dir, f'tmp_{wav_id}.wav')
        # wav_util.cropping_wav_blank(tts_wav_path, tmp_wav_path)
        #
        # # 在音频尾部拼接空白，每个加号500毫秒
        # if blank_count_list[index] > 0:
        #     wav_util.concat_blank_wav(tmp_wav_path, wav_path, blank_count_list[index])
        #     common_util.rmfile(tmp_wav_path)
        # else:
        #     shutil.move(tmp_wav_path, wav_path)

        # 移动为正式片段
        wav_path = os.path.join(filesystem.runtime_wav_dir, f'{wav_id}.wav')
        shutil.move(tts_wav_path, wav_path)

        wav_path_list.append(wav_path)

    # TTS生成成功，返回wav文件路径列表
    return wav_path_list


def get_wav_list(text_list: list, is_assistant: bool) -> list:
    """获取wav列表，失败或超时会抛出异常
    """

    # ####
    # url  >> http://127.0.0.1:$TTS_PORT/getWavList
    # req  -> {text: str, prompt: str?}
    # resp <- wav_list
    # ####

    texts = ''
    for text in text_list:
        if len(text) > 0 and text[-1] not in string.punctuation:
            texts += text
        else:
            texts += (text + '。')

    prompt = '1.wav' if not is_assistant else '2.wav'
    port = const.TTS1_PORT if not is_assistant else const.TTS2_PORT

    url = f'http://127.0.0.1:{port}/getWavList'
    params = {'text': texts, 'prompt': prompt}
    resp = request_util.post(url, params, timeout=90)
    if resp['code'] != 0:
        logging.info(f'文本转语音失败：{port} {resp["msg"]}')
        raise RuntimeError('文本转语音失败')

    logging.info(f'文本转语音成功：{port} {texts}')
    return resp['data']
